/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_dtimer.h                                                                    *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: Services for work with dynamic timers                                           *
 *                                                                                              *
 *  History:    21.01.2009                                                                      *
 *                                                                                              *
 ************************************************************************************************
 */





/************************************************************************************************
 *                                                                                              *
 *                                 D Y N A M I C   T I M E R S                                  *
 *                                                                                              *
 ************************************************************************************************/






#ifdef OS_ENABLE_DTIMERS

/************************************************************************/
/* Check dynamic timer's states                                         */
/************************************************************************/

#define OS_Dtimer_Check(dtimer)      ((dtimer).Flags.bTimeout)       // overflowed
#define OS_Dtimer_IsActive(dtimer)   ((dtimer).Flags.bActive)        // created
#define OS_Dtimer_IsStopped(dtimer)  (!((dtimer).Flags.bRun))        // stopped
#define OS_Dtimer_IsRun(dtimer)      ((dtimer).Flags.bRun)           // run

//------------------------------------------------------------------------------
// Get remaining time of counting
#define OS_Dtimer_Get(dtimer)      (-(dtimer).Timer)



/************************************************************************/
/* Stop, pause and continue counting                                    */
/************************************************************************/

#define OS_Dtimer_Stop(dtimer)          (dtimer).Flags.bRun = 0        // Stop timer (pause it)
#define OS_Dtimer_Pause(dtimer)         (dtimer).Flags.bRun = 0        // Stop timer (pause it)
#define OS_Dtimer_Continue(dtimer)      (dtimer).Flags.bRun = 1        // Continue after pause

#define OS_Dtimer_Break(dtimer)         (dtimer).Flags.bTimeout = 1    //



/************************************************************************/
/* Services for run dynamic timers                                      */
/************************************************************************/

//------------------------------------------------------------------------------
// Start counting with given time. Timeout flag is cleared.

#define OS_Dtimer_Run(dtimer, time)         \
    {                                       \
        (dtimer).Flags.bRun = 0;            \
        (dtimer).Timer = -time;             \
        (dtimer).Flags.bTimeout = 0;        \
        (dtimer).Flags.bRun = 1;            \
    }

//------------------------------------------------------------------------------
// Re-run static timer for time time since last overflow. Timeout flag is cleared.

#define OS_Dtimer_Update(dtimer, time)      \
    {                                       \
        (dtimer).Flags.bRun = 0;            \
        (dtimer).Timer -= time;             \
        (dtimer).Flags.bTimeout = 0;        \
        (dtimer).Flags.bRun = 1;            \
    }

//------------------------------------------------------------------------------
// Increase time of dinamic timer counter. Timeout flag remain unchanged.

#define OS_Dtimer_Add(dtimer, time)         \
    {                                       \
        (dtimer).Flags.bRun = 0;            \
        (dtimer).Timer -= time;             \
        (dtimer).Flags.bRun = 1;            \
    }






/************************************************************************/
/* Wait services                                                        */
/************************************************************************/

//------------------------------------------------------------------------------
// Wait for dynamic timer overflowed

#define OS_Dtimer_Wait(dtimer)                  \
    {                                           \
        _OS_WAIT_EVENT(OS_Dtimer_Check(dtimer));\
    }


//------------------------------------------------------------------------------
// Delay current task using dynamic timer

#define OS_Dtimer_Delay(dtimer, delay)          \
    {                                           \
        OS_Dtimer_Run(dtimer, delay);           \
        OS_Dtimer_Wait(dtimer);                 \
    }









/************************************************************************/
/* REDEFINING OLD NAMES                                                 */
/************************************************************************/

#define OS_CheckPTimer      OS_Dtimer_Check
#define OS_IsPTimerActive   OS_Dtimer_IsActive
#define OS_IsPTimerStopped  OS_Dtimer_IsStopped
#define OS_IsPTimerRun      OS_Dtimer_IsRun
#define OS_GetPTimer        OS_Dtimer_Get
#define OS_StopPTimer       OS_Dtimer_Stop
#define OS_PausePTimer      OS_Dtimer_Pause
#define OS_ContinuePTimer   OS_Dtimer_Continue
#define OS_BreakPTimer      OS_Dtimer_Break
#define OS_RunPTimer        OS_Dtimer_Run
#define OS_UpdatePTimer     OS_Dtimer_Update
#define OS_AddPTimer        OS_Dtimer_Add
#define OS_WaitPTimer       OS_Dtimer_Wait
#define OS_DelayPTimer      OS_Dtimer_Delay
#define OS_CreatePTimer     OS_Dtimer_Create
#define OS_DeletePTimer     OS_Dtimer_Delete
#define OST_PTIMER          OST_DTIMER




#endif  // OS_ENABLE_DTIMERS
















