/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_bsem.h                                                                      *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: Services for work with binary semaphores                                        *
 *                                                                                              *
 *  History:    21.01.2009                                                                      *
 *                                                                                              *
 ************************************************************************************************
 */

/************************************************************************************************
 *                                                                                              *
 *                          B I N A R Y    S E M A P H O R E S                                  *
 *                                                                                              *
 ************************************************************************************************/





#define _OS_BSEM_POS(binsem)        ((binsem) >> _OST_INT_SHIFT)
#define _OS_BSEM_MASK(binsem)       (1 << ((binsem) & _OST_INT_MASK))




//------------------------------------------------------------------------------
// Signal binary semaphore
//
#define OS_Bsem_Set(binsem)        {OS_BSems[_OS_BSEM_POS(binsem)] |= _OS_BSEM_MASK(binsem);}
#define OS_Bsem_Signal(binsem)     OS_Bsem_Set(binsem)



//------------------------------------------------------------------------------
// Reset binary semaphore
//
#define OS_Bsem_Reset(binsem)     {OS_BSems[_OS_BSEM_POS(binsem)] &= ~_OS_BSEM_MASK(binsem);}



//------------------------------------------------------------------------------
// Check for binary semaphore is set
//
#define OS_Bsem_Check(binsem)     (OS_BSems[_OS_BSEM_POS(binsem)] & _OS_BSEM_MASK(binsem))



//------------------------------------------------------------------------------
// Inverce semaphore's value
//
#define OS_Bsem_Switch(binsem)    {OS_BSems[_OS_BSEM_POS(binsem)] ^= _OS_BSEM_MASK(binsem);}




//------------------------------------------------------------------------------
// Wait binary semaphore

#define OS_Bsem_Wait(binsem)                                            \
    {                                                                   \
        _OS_WAIT_EVENT(OS_Bsem_Check(binsem));                          \
        OS_Bsem_Reset(binsem);                                          \
    }



//-------------------------------------------------------------------------------
// Wait binary semaphore. Exit if timeout expired

#define OS_Bsem_Wait_TO(binsem, timeout)                                \
    {                                                                   \
        _OS_WAIT_EVENT_TIMEOUT(OS_Bsem_Check(binsem), timeout);         \
        OS_Bsem_Reset(binsem);                                          \
    }





















