/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_tasks.h                                                                     *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: Services for work with tasks                                                    *
 *                                                                                              *
 *  History:    26.02.2009  Bug fixed: OS_Task_Replace did not reserve created task by fncall   *
 *                          OS_ReserveTask renamed to OS_Task_Reserve                           *
 *                                                                                              *
 ************************************************************************************************
 */



/************************************************************************************************
 *                                                                                              *
 *                                   T A S K S                                                  *
 *                                                                                              *
 ************************************************************************************************/


#define this_task   _OS_CurTask

/************************************************************************/
/*                                                                      */
/* Stop current task                                                    */
/*                                                                      */
/************************************************************************/

#define OS_Task_Stop()   OS_Task_Delete(this_task)


/************************************************************************/
/*                                                                      */
/* Delete task                                                          */
/*                                                                      */
/************************************************************************/

#define OS_Task_Delete(pTask)                                           \
    {                                                                   \
        (pTask)->State.bEnable = 0;                                     \
        if ((&(pTask)) == (&_OS_CurTask))                               \
        {                                                               \
            _OS_Return();                                               \
        }                                                               \
    }


/************************************************************************/
/*                                                                      */
/* Pause task                                                           */
/*                                                                      */
/************************************************************************/

#define OS_Task_Pause(pTask)                                            \
    {                                                                   \
        (pTask)->State.bPaused = 1;                                     \
        if ((&(pTask)) == (&_OS_CurTask))                               \
        {                                                               \
            _OS_Return();                                               \
        }                                                               \
    }

/************************************************************************/
/*                                                                      */
/* Continue paused task                                                 */
/*                                                                      */
/************************************************************************/

#define OS_Task_Continue(pTask)          (pTask)->State.bPaused = 0



/************************************************************************/
/*                                                                      */
/* Replace current task with new (current task will be deleted)         */
/*                                                                      */
/************************************************************************/

//------------------------------------------------------------------------------
#ifdef __OSA18__
//------------------------------------------------------------------------------

    #if OS_CODE_POINTER_SIZE <= 2

        #define _OS_SetIndfWord(value)                                  \
            _postinc = (((OST_CODE_POINTER)value)) & 0xFF;              \
            _postinc = (((OST_CODE_POINTER)value) >> 8) & 0xFF;

    #else
        #define _OS_SetIndfWord(value)                                  \
            _postinc = (((OST_CODE_POINTER)value)) & 0xFF;              \
            _postinc = (((OST_CODE_POINTER)value) >> 8) & 0xFF;         \
            _postinc = (((OST_CODE_POINTER)value) >> 16) & 0xFF;        \
            _postinc = 0;
    #endif

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------




//------------------------------------------------------------------------------
#if     defined(__OSAMCC30__)
//------------------------------------------------------------------------------

    #define OS_Task_Replace(priority, TaskName)                                     \
    {                                                                               \
        OS_Task_Reserve(TaskName);                                                  \
        *((_OST_INT_TYPE*)&OS_state) = priority | 0x48; /* bEnable  bReady */      \
        _OS_CurTask->pTaskPointer = (OST_CODE_POINTER)TaskName;                     \
        _OS_CurTask->State = OS_state;                                              \
        OS_Flags.bTaskReplaced = 1;                                                 \
        _OS_Return();                                                               \
    }

//------------------------------------------------------------------------------
#elif   defined(__OSAMCC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Replace(priority, TaskName)                                     \
    {                                                                               \
        OS_Task_Reserve(TaskName);                                                  \
        _OS_SET_FSR_CUR_TASK();                                                     \
        _postinc = priority | 0x48; /* bEnable  bReady */                          \
        _OS_SetIndfWord(TaskName);                                                  \
        _indf = 0;  /* temp FSR1 */                                                 \
        _OS_Return();                                                               \
    }

//------------------------------------------------------------------------------
#elif   defined(__OSAPICC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Replace(priority, TaskName)                                     \
    {                                                                               \
        OS_Task_Reserve(TaskName);                                                  \
        *((NEAR _OST_INT_TYPE*)&OS_state) = priority | 0x48; /* bEnable  bReady */ \
        _OS_SET_FSR_CUR_TASK();                                                     \
        _postinc = priority | 0x48; /* bEnable  bReady */                          \
        _OS_SetIndfWord(TaskName);                                                  \
        _OS_Return();                                                               \
    }

//------------------------------------------------------------------------------
#elif   defined(__OSAPICC12__)
//------------------------------------------------------------------------------

    #define OS_Task_Replace(priority, TaskName)                                     \
    {                                                                               \
        OS_Task_Reserve(TaskName);                                                  \
        *((_OST_INT_TYPE*)&OS_state) = priority | 0x48; /* bEnable  bReady */      \
        _fsr = (_FSR_TYPE*)_OS_CurTask;                                             \
        _indf = priority | 0x48; /* bEnable  bReady */                             \
        _fsr++;                                                                     \
        _indf = ((OST_CODE_POINTER)TaskName) & 0xFF;                                \
        _fsr = 0;                                                                   \
        _OS_Return();                                                               \
    }


//------------------------------------------------------------------------------
#elif   defined(__OSAPICC16__)
//------------------------------------------------------------------------------

    #define OS_Task_Replace(priority, TaskName)                                     \
    {                                                                               \
        OS_Task_Reserve(TaskName);                                                  \
        _OS_SET_FSR_CUR_TASK();                                                     \
        _indf = priority | 0x48; /* bEnable  bReady */                             \
        _fsr++;                                                                     \
        _indf = ((OST_CODE_POINTER)TaskName) & 0xFF;                                \
        _fsr++;                                                                     \
        _indf = (((OST_CODE_POINTER)TaskName) >> 8) & 0xFF;                         \
        _OS_Return();                                                               \
    }


//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------






/****************************************************************************/
/*                                                                          */
/* Create task                                                              */
/*                                                                          */
/****************************************************************************/


//------------------------------------------------------------------------------
#if defined(__OSAMCC30__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName) _OS_Task_Create(priority, (OST_CODE_POINTER)TaskName)

//------------------------------------------------------------------------------
#elif defined (__OSAMCC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName) _OS_Task_Create(priority, (OST_CODE_POINTER)TaskName)

//------------------------------------------------------------------------------
#elif defined(__OSACCS18__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName)                          \
        {                                                               \
            _OS_Task_Create(priority, (OST_CODE_POINTER)TaskName);      \
            if (_pcl & 1) TaskName();                                   \
        }

//------------------------------------------------------------------------------
#elif defined(__OSAPICC12__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName)                          \
        {                                                               \
            asm("\t global _main       ");                              \
            asm("\t global _"#TaskName);                                \
            asm("\t fncall _main, _"#TaskName);                         \
            asm("_OS_Add_TaskAddr   _"#TaskName);                       \
            _OS_Task_Create(priority, _OS_temp);                        \
        }

//------------------------------------------------------------------------------
#elif defined(__OSAPICC16__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName)                          \
        {                                                               \
            asm("\t global _main       ");                              \
            asm("\t global _" #TaskName);                               \
            asm("\t fncall _main,_" #TaskName);                         \
            _OS_Task_Create(priority, (OST_CODE_POINTER)TaskName);      \
        }
//------------------------------------------------------------------------------
#elif defined(__OSAPICC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Create(priority, TaskName)                          \
        {                                                               \
            asm("\t global _main       ");                              \
            asm("\t global _" #TaskName);                               \
            asm("\t fncall _main,_" #TaskName);                         \
            _OS_Task_Create(priority, (OST_CODE_POINTER)TaskName);      \
        }
//------------------------------------------------------------------------------
#else
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------






/************************************************************************/
/*                                                                      */
/* Create task by indirrect address                                     */
/*                                                                      */
/************************************************************************/

#define OS_Task_CreateP(priority, TaskAddr)      _OS_Task_Create(priority, TaskAddr)


// If task created indirrectly then it is nessesary to insert in
// the main() servise OS_ReserveTasks

//------------------------------------------------------------------------------
#if defined(__OSAMCC30__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)

//------------------------------------------------------------------------------
#elif defined (__OSAMCC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)

//------------------------------------------------------------------------------
#elif defined(__OSACCS18__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)

//------------------------------------------------------------------------------
#elif defined(__OSAPICC12__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)                                       \
        {                                                                   \
            asm("\t global _main       ");                                  \
            asm("\t global _" #TaskName);                                   \
            asm("\t fncall _main,_" #TaskName);                             \
        }

//------------------------------------------------------------------------------
#elif defined(__OSAPICC16__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)                                       \
        {                                                                   \
            asm("\t global _main       ");                                  \
            asm("\t global _" #TaskName);                                   \
            asm("\t fncall _main,_" #TaskName);                             \
        }

//------------------------------------------------------------------------------
#elif defined(__OSAPICC18__)
//------------------------------------------------------------------------------

    #define OS_Task_Reserve(TaskName)                                       \
        {                                                                   \
            asm("\t global _main       ");                                  \
            asm("\t global _" #TaskName);                                   \
            asm("\t fncall _main,_" #TaskName);                             \
        }

//------------------------------------------------------------------------------
#else
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------


/************************************************************************/
/*                                                                      */
/* Get/set current task priority                                        */
/*                                                                      */
/************************************************************************/

#define OS_Task_GetPriority(pTask)                   (pTask)->State.Priority
#define OS_Task_SetPriority(pTask, new_priority)     (pTask)->State.Priority = new_priority


/************************************************************************/
/*                                                                      */
/* Get current task's descriptor address                                */
/*                                                                      */
/************************************************************************/

#define OS_GetCurTask()                 this_task


/************************************************************************/
/*                                                                      */
/* Check for task is paused                                             */
/*                                                                      */
/************************************************************************/

#define OS_Task_IsPaused(pTask)          (pTask)->State.bPaused



















