/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_stimer.h                                                                    *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: Services for work with static timers                                            *
 *                                                                                              *
 *  History:    21.01.2009                                                                      *
 *                                                                                              *
 ************************************************************************************************
 */



/************************************************************************************************
 *                                                                                              *
 *                         S T A T I C   T I M E R S   S E R V I C E S                          *
 *                                                                                              *
 ************************************************************************************************/



//------------------------------------------------------------------------------
#ifdef OS_ENABLE_STIMERS
//------------------------------------------------------------------------------


#if     OS_STIMER_SIZE == 1
    #define OS_STIMER_RUN   0x80
#elif   OS_STIMER_SIZE == 2
    #define OS_STIMER_RUN   0x8000
#elif   OS_STIMER_SIZE == 4
    #define OS_STIMER_RUN   0x80000000
#endif



/************************************************************************/
/* Check timer's states                                                 */
/************************************************************************/

#define OS_Stimer_Check(stimer_id)   (!(OS_STimers[stimer_id] & OS_STIMER_RUN))
#define OS_Stimer_IsRun(stimer_id)   (OS_STimers[stimer_id] & OS_STIMER_RUN)


/************************************************************************/
/* Run timer for counting                                               */
/************************************************************************/

#define OS_Stimer_Run(stimer_id, time)              \
    {                                               \
        OS_Stimer_Stop(stimer_id);                  \
        OS_STimers[stimer_id] = -(time);            \
    }

    // At the end of this macro we can't set bit Run since before we do it,
    // flag can became clear in interrupt.


/************************************************************************/
/* Stop/pause timer                                                     */
/************************************************************************/

#define OS_Stimer_Stop(stimer_id)    OS_STimers[stimer_id] &= ~OS_STIMER_RUN
#define OS_Stimer_Pause(stimer_id)    OS_STimers[stimer_id] &= ~OS_STIMER_RUN


/************************************************************************/
/* Get remaining time of counting                                       */
/* It is nessesary to disable interrupt ehile reading STimer value      */
/************************************************************************/

#define OS_Stimer_Get(stimer_id)    (OS_Stimer_Check(stimer_id)? 0:(-OS_STimers[stimer_id]))


/************************************************************************/
/* Continue counting after pause                                        */
/************************************************************************/

#define OS_Stimer_Continue(stimer_id)    OS_STimers[stimer_id] |= OS_STIMER_RUN


/************************************************************************/
/* Wait for timer overflow                                              */
/************************************************************************/

#define OS_Stimer_Wait(stimer_id)                   \
    {                                               \
        _OS_WAIT_EVENT(OS_Stimer_Check(stimer_id)); \
    }


/************************************************************************/
/* Delay current task using static timer                                */
/************************************************************************/

#define OS_Stimer_Delay(stimer_id, delay)           \
    {                                               \
        OS_Stimer_Run(stimer_id, delay);            \
        OS_Stimer_Wait(stimer_id);                  \
    }



//------------------------------------------------------------------------------
#endif  // OS_ENABLE_STIMERS
//------------------------------------------------------------------------------




















